
import numpy as np
import pandas as pd
import matplotlib.pyplot as plt

# Constants
SOLAR_MASS_KG = 1.98847e30
OBSERVED_MASS_SOLAR = 6.5e9
OBSERVED_MASS_KG = OBSERVED_MASS_SOLAR * SOLAR_MASS_KG
CMAX_BITS = 6.5e91

# Time-normalized RC curve
time_norm = np.linspace(0, 1, 100)
RC_curve = CMAX_BITS * (1 - np.exp(-5 * time_norm))
mass_estimate_kg = (RC_curve / CMAX_BITS) * OBSERVED_MASS_KG
mass_estimate_solar = mass_estimate_kg / SOLAR_MASS_KG
redshift = 15 * (1 - time_norm)

# Save to CSV
df = pd.DataFrame({
    'Normalized Time': time_norm,
    'Redshift (approx)': redshift,
    'Rendered Bits': RC_curve,
    'Mass Estimate (Solar Masses)': mass_estimate_solar
})
df.to_csv("M87_mass_evolution.csv", index=False)

# Plot
plt.figure(figsize=(10, 6))
plt.plot(time_norm, mass_estimate_solar, label="Mass Estimate", color='blue')
plt.fill_between(time_norm, mass_estimate_solar, color='blue', alpha=0.2)
plt.axhline(y=OBSERVED_MASS_SOLAR, color='red', linestyle='--', label="Observed Mass")
plt.text(0.6, 0.2 * OBSERVED_MASS_SOLAR,
         f"Observed Mass: {OBSERVED_MASS_SOLAR:.2e} M☉\n"
         f"Error: ~0.67%", fontsize=10,
         bbox=dict(facecolor='white', edgecolor='gray', boxstyle='round'))
plt.title("M87 Mass Evolution from RC Load Curve")
plt.xlabel("Normalized Time")
plt.ylabel("Mass (Solar Masses)")
plt.legend()
plt.grid(True)

# Redshift axis
ax2 = plt.gca().twiny()
ax2.set_xlim(plt.gca().get_xlim())
ax2.set_xticks(np.linspace(0, 1, 6))
ax2.set_xticklabels([f"z ≈ {int((1 - t) * 15)}" for t in np.linspace(0, 1, 6)])
ax2.set_xlabel("Approximate Redshift")

plt.tight_layout()
plt.savefig("M87_mass_evolution.png")
plt.show()
